
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   MSG1001.C - Message 1001 Processing
// 
//
//   DESCRIPTION
//
//   Functions to decode, build, and display Zodiac 1001 messages.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\MSG1001.C_V  $
//   
//      Rev 1.15   09 Nov 1998 10:47:42   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler
//   
//      Rev 1.5   Jul 09 1997 10:36:12   COLEJ
//    
//   
//      Rev 1.4   Feb 12 1997 16:05:16   COLEJ
//    
//   
//      Rev 1.3   Aug 26 1996 18:42:52   COLEJ
//    
//   
//      Rev 1.2   Aug 19 1996 17:12:02   COLEJ
//    
//   
//      Rev 1.1   Aug 14 1996 09:17:00   COLEJ
//   Deleted UTCNSECS field and processing, and moved up other fields.
//   Changed GPS field to display GPSSEC.
//   Added msg1108 coditional processing of UTC TIME
//   
//      Rev 1.0   13 May 1996 14:52:40   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <graph.h>
#include <string.h>
#include <math.h>

#include "gentypes.h"
#include "gencons.h"
#include "ztypes.h"
#include "zcons.h"   
#include "util.h" 
#include "labmon.h"
#include "display.h"
 
extern unsigned long CntNav, CntPnt;

void show_filter(short); 

////////////////////////////////////////////////////////////////////////////////
//   
//   Decode a Zodiac 1001 message from the ZMsgBuf.
//
void Dec1001(tMSGBUF *ZMsgBuf, tMSG1001 *Msg)
{  
   GetBit(&SolutionStatusBits.InvalidAltUsed, ZMsgBuf, 10, 0);  // 1=true
   GetBit(&SolutionStatusBits.InvalidNoDGPS , ZMsgBuf, 10, 1);  // 1=true
   GetBit(&SolutionStatusBits.InvalidNumTrk , ZMsgBuf, 10, 2);  // 1=true 
   GetBit(&SolutionStatusBits.InvalidExcEHPE, ZMsgBuf, 10, 3);  // 1=true
   GetBit(&SolutionStatusBits.InvalidExcEVPE, ZMsgBuf, 10, 4);  // 1=true

   GetBit(&SolutionStatusBits.TypePropagated, ZMsgBuf, 11, 0);  // 1=propagated
   GetBit(&SolutionStatusBits.TypeAltUsed   , ZMsgBuf, 11, 1);  // 1=altitude used
   GetBit(&SolutionStatusBits.TypeDGPS      , ZMsgBuf, 11, 2);  // 1=DGPS mode
}                                      
                                     
////////////////////////////////////////////////////////////////////////////////
//   
//   Build a Zodiac 1001 message in the ZMsgBuf. 
//
void Bld1001(tMSGBUF *ZMsgBuf, tMSG1001 *Msg)
{
   // ack flags assumed preset

   PutShort  (0x81FF                       , ZMsgBuf,  1);      // message preamble
   PutShort  (1001                         , ZMsgBuf,  2);      // message number
   PutShort  (47                           , ZMsgBuf,  3);      // message length 
 //PutShort  (CheckSum(ZMsgBuf[0], 4)       , ZMsgBuf, 5);      // header check sum 
   
   PutBit(SolutionStatusBits.InvalidAltUsed, ZMsgBuf, 10, 0);  // 1=true
   PutBit(SolutionStatusBits.InvalidNoDGPS , ZMsgBuf, 10, 1);  // 1=true
   PutBit(SolutionStatusBits.InvalidNumTrk , ZMsgBuf, 10, 2);  // 1=true 
   PutBit(SolutionStatusBits.InvalidExcEHPE, ZMsgBuf, 10, 3);  // 1=true
   PutBit(SolutionStatusBits.InvalidExcEVPE, ZMsgBuf, 10, 4);  // 1=true

   PutBit(SolutionStatusBits.TypePropagated, ZMsgBuf, 11, 0);  // 1=propagated
   PutBit(SolutionStatusBits.TypeAltUsed   , ZMsgBuf, 11, 1);  // 1=altitude used
   PutBit(SolutionStatusBits.TypeDGPS      , ZMsgBuf, 11, 2);  // 1=DGPS mode
    
 //PutShort  (CheckSum(ZMsgBuf, 6, 48)      , ZMsgBuf, 53);    // data check sum 
}

////////////////////////////////////////////////////////////////////////////////
//   
//   Display a Zodiac 1001 message from the ZMsgBuf.
//
void Show1001(tMSG1001 *Msg)
{
   char    Buf[80] = "";
   
   
   char    *DayName[7] ={"SUN", "MON", "TUE", "WED", "THU", "FRI", "SAT"}; 
   char    UTCDayMonthYear[15],UTCHourMinSec[15]; 
   char    DayOfWeek[5];  
   char    NavigationMode[6], SolutionInvalid[6], SolutionType[4];      
//   short   GPSHours, GPSMinutes, GPSSeconds;
   double  GPSTimeSeconds,GPSTimeNSeconds;
   unsigned long  tempsecs;
   unsigned short gpsprecision = 7;  // used to calculate precision field of gpssec output
  
   extern unsigned short StatMask;   
   extern float  maxpdop, maxhdop, maxvdop;
   extern short  minsats;
   extern short  filter_on; 
   extern short  xtract;  
   extern char   xtractid[4];
   extern tBOOL  MSG1108ON; 
   
   extern FILE   *stream3; 
   
   // convert values
      
   // perform special processing for output
 
   // navigation validity  
   strcpy(SolutionInvalid, "     ");
   if(SolutionStatusBits.InvalidAltUsed ) SolutionInvalid[0] = 'A';
   if(SolutionStatusBits.InvalidNoDGPS  ) SolutionInvalid[1] = 'D';
   if(SolutionStatusBits.InvalidNumTrk  ) SolutionInvalid[2] = 'N';
   if(SolutionStatusBits.InvalidExcEHPE ) SolutionInvalid[3] = 'H';
   if(SolutionStatusBits.InvalidExcEVPE ) SolutionInvalid[4] = 'V';

   // navigation solution type and mode
   strcpy(SolutionType, "   "); 
   strcpy(NavigationMode, "  NAV");  
   
   if(SolutionStatusBits.TypePropagated ){
      SolutionType[0]    = 'P';
   }
   if(SolutionStatusBits.TypeAltUsed    ){
      SolutionType[1]    = 'A';
   }    
   if(SolutionStatusBits.TypeDGPS       ){
      SolutionType[2]    = 'D'; 
      strcpy(NavigationMode, " DGPS");
   }       
   
   // navigation mode
   if(strcmp(SolutionInvalid, "     ") != 0){
      strcpy(NavigationMode, "  ACQ");
   }
   
   // time values  -  only display gps seconds jc 7-29-96
//   GPSHours    = (short) (Msg->GPSTimeSeconds / 3600.f);
//   TimeTemp    = Msg->GPSTimeSeconds - GPSHours * 3600.f;
//   GPSMinutes  = (short) (TimeTemp  / 60.f);
//   TimeTemp    = TimeTemp - GPSMinutes * 60.f;
//   GPSSeconds  = (short) TimeTemp;


   // calculate gpssec field
   tempsecs = Msg->GPSTimeSeconds;
   while( tempsecs /= 10 ){ gpsprecision--; };
   // get rid of roundoff for nano seconds
   GPSTimeNSeconds = (double) ( Msg->GPSTimeNanoseconds / (unsigned long) pow( 10.0, (9.0 - (double) gpsprecision) ) );
   GPSTimeNSeconds *= pow( 10.0, (9.0 - (double) gpsprecision) );
   GPSTimeSeconds = Msg->GPSTimeSeconds + GPSTimeNSeconds / 1E+09;


   sprintf(DayOfWeek,"%s",
           DayName[(short)(Msg->GPSTimeSeconds / 86400)]);
   sprintf(UTCDayMonthYear,"%02d/%02d/%02d",
           Msg->UTCMonth,Msg->UTCDay,(Msg->UTCYear % 100));
   sprintf(UTCHourMinSec,"%02d:%02d:%02d",
           Msg->UTCHours,Msg->UTCMinutes,Msg->UTCSeconds);  
//   sprintf(GPSHourMinSecNsec,"%3d:%02d:%02d.000",
//           GPSHours,GPSMinutes,GPSSeconds);  
            
   // filter the data using binary mode parameters      
   CntPnt++; 
   
   if(((StatMask & Msg->SolutionValidity) == 0) &&      
      (Msg->NumMeasUsed   >= (unsigned short) minsats)           ){       
               
      filter_on = 0; 
      CntNav++;
             
      // write extracted data to a file
      if(xtract){
         if(strcmp(xtractid,"PV") == 0){     
            fprintf(stream3, 
                    "%16.9f \t%10.1f \t%10.1f \t%10.1f \t%8.2f \t%8.2f \t%8.2f\n",      
                    GPSTimeSeconds, 
                    Msg->ECEFPositionX /1E+02,
                    Msg->ECEFPositionY /1E+02,
                    Msg->ECEFPositionZ /1E+02,
                    Msg->ECEFVelocityX /1E+02,
                    Msg->ECEFVelocityY /1E+02,
                    Msg->ECEFVelocityZ /1E+02); 
         }
         else if(strcmp(xtractid,"COV") == 0){     
            fprintf(stream3, 
            "%16.9f \t%9.2G \t%9.2G \t%9.2G \t%9.2G \t%10.0f \t%9.2G \t%10.0f \t%9.2G\n",      
                    GPSTimeSeconds, 
                    Msg->EHPE             /1E+02,
                    Msg->EVPE             /1E+02,
                    Msg->EHVE             /1E+02,
                    Msg->ETE              /1E+02,
                    Msg->ClockBias        /1E+02,
                    Msg->ClockBiasStdDev  /1E+02,
                    Msg->ClockDrift       /1E+02,
                    Msg->ClockDriftStdDev /1E+02);
         } 
      }                
   }      
   else{      
      filter_on = 1;      
   }      
    
   // indicate when filtering is occurring 
   show_filter(filter_on);


   // output the data items   
   sprintf(Buf,"%05d"  ,Msg->SequenceNumber          );ShowText(Buf,TIMR+ 6,TIMC+ 4);
   sprintf(Buf,"%05d"  ,Msg->MeasSequenceNumber      );ShowText(Buf,TIMR+ 6,TIMC+12);
   sprintf(Buf,"%05s"  ,NavigationMode               );ShowText(Buf,MODR   ,MODC+ 5);
   sprintf(Buf,"%05s"  ,SolutionInvalid              );ShowText(Buf,MODR+ 1,MODC+ 5);
   sprintf(Buf,"%03s"  ,SolutionType                 );ShowText(Buf,MODR+ 2,MODC+ 7);
   sprintf(Buf,"%4d"   ,Msg->NumMeasUsed             );ShowText(Buf,DOPR- 2,DOPC+ 7);
   sprintf(Buf,"%4d "  ,Msg->GPSWeek                 );ShowText(Buf,TIMR+ 4,TIMC+ 5);
   sprintf(Buf,"%9.*f" ,gpsprecision,GPSTimeSeconds  );ShowText(Buf,TIMR+ 2,TIMC+ 8);
//   sprintf(Buf,"%s"    ,GPSHourMinSecNsec            );ShowText(Buf,TIMR+ 3,TIMC+ 4);
   sprintf(Buf,"%09ld" ,Msg->GPSTimeNanoseconds      );ShowText(Buf,TIMR+ 3,TIMC+ 8);
   sprintf(Buf,"%s"    ,DayOfWeek                    );ShowText(Buf,TIMR+ 4,TIMC+14);
   sprintf(Buf,"%s"    ,UTCDayMonthYear              );ShowText(Buf,TIMR   ,TIMC+ 9);                 
   if( !MSG1108ON ){sprintf(Buf,"  %s",UTCHourMinSec );ShowText(Buf,TIMR+ 1,TIMC+ 7);}
//   sprintf(Buf,"%09ld" ,Msg->UTCNanoseconds          );ShowText(Buf,TIMR+ 2,TIMC+ 8);   
   sprintf(Buf,"%10.1f",Msg->ECEFPositionX     /1E+02);ShowText(Buf,POSR   ,POSC+ 6);
   sprintf(Buf,"%10.1f",Msg->ECEFPositionY     /1E+02);ShowText(Buf,POSR+ 1,POSC+ 6);
   sprintf(Buf,"%10.1f",Msg->ECEFPositionZ     /1E+02);ShowText(Buf,POSR+ 2,POSC+ 6);  
   sprintf(Buf,"%6.2f" ,Msg->ECEFVelocityX     /1E+02);ShowText(Buf,VELR   ,VELC+ 4); 
   sprintf(Buf,"%6.2f" ,Msg->ECEFVelocityY     /1E+02);ShowText(Buf,VELR+ 1,VELC+ 4); 
   sprintf(Buf,"%6.2f" ,Msg->ECEFVelocityZ     /1E+02);ShowText(Buf,VELR+ 2,VELC+ 4); 
   sprintf(Buf,"%3d"   ,Msg->MapDatum                );ShowText(Buf,GEOR+ 1,GEOC+ 8); 
   sprintf(Buf,"%9.2G" ,Msg->EHPE              /1E+02);ShowText(Buf,HERR   ,HERC+ 5); 
   sprintf(Buf,"%9.2G" ,Msg->EVPE              /1E+02);ShowText(Buf,HERR+ 1,HERC+ 5); 
   sprintf(Buf,"%9.2G" ,Msg->ETE               /1E+02);ShowText(Buf,HERR+ 2,HERC+ 5); 
   sprintf(Buf,"%9.2G" ,Msg->EHVE              /1E+02);ShowText(Buf,HERR+ 3,HERC+ 5); 
   sprintf(Buf,"%10.0f",Msg->ClockBias         /1E+02);ShowText(Buf,CBDR   ,CBDC+ 2); 
   sprintf(Buf,"%9.2G" ,Msg->ClockBiasStdDev   /1E+02);ShowText(Buf,CBDR+ 1,CBDC+ 3); 
   sprintf(Buf,"%10.0f",Msg->ClockDrift        /1E+02);ShowText(Buf,CBDR+ 2,CBDC+ 2); 
   sprintf(Buf,"%9.2G" ,Msg->ClockDriftStdDev  /1E+02);ShowText(Buf,CBDR+ 3,CBDC+ 3);                                    
}                                                 
                                      
                                           
